function cds_new = update_model_price_cds(cds, discounts_IMM, maturities, start_date_num, end_date_num, liq_prem_cds, ajd_common, y0)
% --------------------------------------------------------------------------------------------------
% Update the model implied coupon rate for a CDS.
% --------------------------------------------------------------------------------------------------
% cds             ... structure containing credit default swap time series (see 'all_steps_in_a_row.m')
% discounts_IMM   ... structure with discount curves matching IMM dates
% maturities      ... which maturities to update (indicator vector), default: update all
% liq_prem_cds    ... (time series of) liquidity premium for CDS vs. index
% ajd_common      ... dynamics of common factor
% y0              ... initial value(s) of common factor
% --------------------------------------------------------------------------------------------------
% sample call: update_model_price_cds(cds_test, discounts_IMM)
% --------------------------------------------------------------------------------------------------

% Set default parameter values
if (nargin <= 2)
    maturities = ones(length(cds.T), 1);
end
if (nargin <= 3)
    if (length(cds.dates{1}) > 0)
        start_date_num = cds.dates{1}(1);
        end_date_num = cds.dates{1}(end);
    else
        cds_new = cds;
        return;
    end
end

% Calculate risk-neutral survival and default probability for each (future) coupon payment date
default_probs = default_probs_cds2(cds, maturities, start_date_num, end_date_num, ajd_common, y0);

for i=1:length(maturities)
    if (maturities(i) == 1)
        % Allocate memory for model-implied prices
        if (~isfield(cds, 'model_price'))
            cds.model_price{i} = zeros(length(cds.dates{i}), 1);
        end
        
        % Determine relevant date range
        start_pos = find(cds.dates{i} >= start_date_num, 1, 'first');
        end_pos = find(cds.dates{i} <= end_date_num, 1, 'last');
        used_range = start_pos:end_pos;
        if (isempty(used_range))
            continue;
        end
        
        % Extract relevant discount curves
        [discounts, discounts_inbetween] = get_discounts(cds.dates{i}(used_range), discounts_IMM, cds.T{i}(used_range));
        
        % Solve for model implied coupon rate
        npv_fixed_leg = value_fixed_leg_CDS(cds.dates{i}(used_range), cds.T{i}(used_range), cds.market_price{i}(used_range), ...
                                            cds.LGD, discounts, discounts_inbetween, default_probs, liq_prem_cds);
        npv_default_leg = value_default_leg_CDS(cds.dates{i}(used_range), cds.T{i}(used_range), cds.market_price{i}(used_range), ...
                                                cds.LGD, discounts, discounts_inbetween, default_probs);
        c = npv_default_leg ./ npv_fixed_leg .* cds.market_price{i}(used_range);

        % Update CDS with model-implied price
        if ((~isfield(cds, 'model_price')) | (~iscell(cds.model_price)))
            cds.model_price = {[] [] [] []};
        end
        if (length(cds.model_price{i}) == 0)
            cds.model_price{i} = zeros(length(cds.dates{i}), 1);
        end
        cds.model_price{i}(used_range,:) = c;
     end
end
cds_new = cds;
